<?php
/*  VC Card Maker – single file
 *  Drop into /public_html/vc/cardmaker.php
 *  Generates: /public_html/vc/<slug>/index.html + <First_Last>.vcf
 */

function h($s){ return htmlspecialchars($s ?? "", ENT_QUOTES, 'UTF-8'); }
function cleanPhone($p){
  $p = preg_replace('/\s+/', '', $p ?? '');
  if($p && $p[0] !== '+') $p = '+91'.$p; // default to +91 if missing country
  return $p;
}
function slugify($txt){
  $s = strtolower(trim($txt ?? ''));
  $s = preg_replace('/[^a-z0-9\-]+/','-', $s);
  $s = preg_replace('/-+/','-', $s);
  return trim($s, '-') ?: 'contact';
}
function buildVCard($d){
  $lines = [
    "BEGIN:VCARD",
    "VERSION:3.0",
    "N:{$d['last_name']};{$d['first_name']};;;",
    "FN:{$d['first_name']} {$d['last_name']}",
    "TITLE:{$d['title']}",
    "ORG:{$d['org']}",
  ];
  if(!empty($d['phone1'])) $lines[] = "TEL;TYPE=CELL,VOICE:{$d['phone1']}";
  if(!empty($d['phone2'])) $lines[] = "TEL;TYPE=CELL,VOICE:{$d['phone2']}";
  if(!empty($d['whatsapp'])) $lines[] = "TEL;TYPE=CELL,VOICE,X-WhatsApp:{$d['whatsapp']}";
  if(!empty($d['email1'])) $lines[] = "EMAIL;TYPE=WORK,INTERNET:{$d['email1']}";
  if(!empty($d['email2'])) $lines[] = "EMAIL;TYPE=WORK,INTERNET:{$d['email2']}";
  $adr = "ADR;TYPE=WORK:;;{$d['street']};{$d['city']};{$d['region']};{$d['postal']};{$d['country']}";
  $lines[] = $adr;
  if(!empty($d['website'])) $lines[] = "URL:{$d['website']}";
  if(!empty($d['linkedin'])) $lines[] = "URL:{$d['linkedin']}";
  if(!empty($d['twitter'])) $lines[] = "URL:{$d['twitter']}";
  if(!empty($d['instagram'])) $lines[] = "URL:{$d['instagram']}";
  if(!empty($d['facebook'])) $lines[] = "URL:{$d['facebook']}";
  $lines[] = "END:VCARD";
  return implode("\n", $lines);
}

function buildHtml($d, $vcardFilename){
  $qr = urlencode($d['vcard_raw']);
  $phone1 = $d['phone1_display'];
  $phone2 = $d['phone2_display'];
  $whatsapp = $d['whatsapp_display'];
  $logo = h($d['logo']);
  $brand = h($d['brand']);
  $urlStub = h($d['public_url']);
  $website = h($d['website']);
  $mapQ = urlencode("{$d['street']}, {$d['city']} {$d['postal']}");

  return <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>{$d['first_name']} {$d['last_name']} — {$d['org']} vCard</title>
<meta name="description" content="Save the contact card for {$d['first_name']} {$d['last_name']}, {$d['title']} at {$d['org']}">
<meta property="og:title" content="{$d['first_name']} {$d['last_name']} — vCard">
<meta property="og:description" content="Tap to call, email, or download the contact card (.vcf).">
<meta property="og:type" content="website">
<meta property="og:url" content="{$urlStub}">
<meta name="theme-color" content="{$brand}">
<link rel="preload" as="image" href="{$logo}">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
:root{ --brand:{$brand}; --brand2:#22c55e; --ink:#0b1220; --muted:#667085; --card:#fff; --bg:#f6fbf7; --r:22px; }
*{box-sizing:border-box}
body{margin:0;font-family:Inter,system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial;color:var(--ink);
  background:radial-gradient(1200px 600px at 10% -10%, #d1fade 0%, transparent 60%),
             radial-gradient(1200px 600px at 110% 10%, #e8fff0 0%, transparent 60%),
             linear-gradient(180deg,#f9fffb,var(--bg));
  min-height:100vh; display:flex; align-items:center; justify-content:center; padding:28px;}
.wrap{width:100%; max-width:900px}
header.brand{display:flex;align-items:center;gap:14px;justify-content:center;margin-bottom:16px}
header.brand img{height:42px;width:auto}
header.brand .word{font-weight:800; letter-spacing:.4px; color:#0b3b22}
.hero{background:conic-gradient(from 180deg at 70% -20%, #c9f7d4, #ebffef); border:1px solid #e6f5ea; border-radius:var(--r); padding:18px; margin-bottom:18px; box-shadow:0 8px 30px rgba(2,6,23,.06)}
.hero .bar{height:84px;border-radius:16px;background:linear-gradient(90deg,var(--brand),var(--brand2));display:flex;align-items:center;justify-content:space-between;padding:0 18px;color:#fff}
.hero .title{font-weight:800;font-size:clamp(18px,3.2vw,22px)}
.hero .cta{display:inline-flex;align-items:center;gap:10px;background:#fff;color:var(--brand);padding:10px 14px;border-radius:12px;font-weight:700;text-decoration:none}
.grid{display:grid;grid-template-columns:1.2fr .8fr;gap:22px;background:var(--card);border-radius:var(--r);border:1px solid #e6ebf0;padding:26px;box-shadow:0 10px 36px rgba(2,6,23,.08)}
@media (max-width:820px){.grid{grid-template-columns:1fr}}
h1{margin:0 0 6px 0;font-size:clamp(24px,4.6vw,34px)}
.role{color:var(--muted);margin:0 0 16px}
.social-media{display:flex;gap:12px;margin:16px 0 20px 0;flex-wrap:wrap}
.social-icon{display:inline-flex;align-items:center;justify-content:center;width:36px;height:36px;border-radius:50%;text-decoration:none;font-size:18px;transition:.2s ease;color:#fff}
.social-icon:hover{transform:translateY(-2px);box-shadow:0 4px 12px rgba(0,0,0,.15)}
.social-icon.linkedin{background:#0077b5}
.social-icon.twitter{background:#1da1f2}
.social-icon.instagram{background:linear-gradient(45deg,#f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%)}
.social-icon.facebook{background:#1877f2}
.org{display:inline-flex;align-items:center;gap:10px;padding:6px 10px;background:#ecfdf5;color:var(--brand);border-radius:999px;font-weight:700;font-size:14px;margin-bottom:14px}
.org img{height:18px;width:auto}
.list{display:grid;gap:10px;margin-top:12px}
.item a{text-decoration:none;color:var(--ink);display:flex;align-items:center;gap:12px;padding:12px 14px;border:1px solid #e5e7eb;border-radius:14px;transition:.18s ease;background:#fff}
.item a:hover{transform:translateY(-1px);box-shadow:0 10px 22px rgba(2,6,23,.07)}
.item a[href*="wa.me"]{border-color:#25d366;background:#f0fff4}
.item a[href*="wa.me"]:hover{background:#e6fffa;box-shadow:0 10px 22px rgba(37,211,102,.15)}
.badge{font-size:12px;color:#64748b}
.aside{display:grid;gap:16px}
.qr{display:flex;align-items:center;justify-content:center;background:#fff;border:1px dashed #d1d5db;border-radius:18px;padding:18px}
.qr img{width:100%;max-width:220px;height:auto;display:block}
.download{display:flex;flex-direction:column;gap:10px;background:#f5fff8;border:1px solid #d8f1e1;padding:16px;border-radius:16px}
.btn{display:inline-flex;align-items:center;gap:10px;background:var(--brand);color:#fff;padding:12px 16px;border-radius:12px;text-decoration:none;font-weight:700;border:1px solid #0e8d3c}
.btn.secondary{background:#fff;color:var(--brand);border-color:#b7e3c6}
.btn[href*="wa.me"]{background:#25d366;color:#fff;border-color:#22c55e}
.btn[href*="wa.me"]:hover{background:#22c55e}
footer{text-align:center;color:#6b7280;font-size:13px;margin-top:14px}
</style>
</head>
<body>
  <div class="wrap">
    <header class="brand">
      <img src="{$logo}" alt="{$d['org']} logo"><div class="word"> </div>
    </header>
    <!-- <section class="hero">
      <div class="bar">
        <div class="title">Digital Contact Card</div>
        <a class="cta" href="./{$vcardFilename}" download>⬇️ Save vCard</a>
      </div>
    </section> -->
    <section class="grid">
      <div>
        <span class="org"><img src="{$logo}" alt="">  </span>
        <h1>{$d['first_name']} {$d['last_name']}</h1>
        <p class="role">{$d['title']}</p>
        <div class="social-media">
          PLACEHOLDER_SOCIAL_ICONS
        </div>
        <div class="list">
          <!-- phones -->
          {$phone1}
          {$phone2}
          <!-- WhatsApp -->
          {$whatsapp}
          <!-- emails -->
          <div class="item"><a href="mailto:{$d['email1']}">✉️ {$d['email1']}</a></div>
          PLACEHOLDER_EMAIL2
          <div class="item"><a href="https://maps.google.com/?q={$mapQ}" target="_blank" rel="noopener">📍 {$d['street']}, {$d['city']} {$d['postal']}</a></div>
          <div class="item"><a href="{$website}" target="_blank" rel="noopener">🌐 {$website}</a></div>
        </div>
      </div>
      <aside class="aside">
        <div class="qr"><img src="https://api.qrserver.com/v1/create-qr-code/?size=360x360&data={$qr}" alt="QR code for vCard"></div>
        <div class="download">
          <div style="font-weight:700">Save this contact</div>
          <a class="btn" href="./{$vcardFilename}" download>⬇️ Download vCard (.vcf)</a>
          <a class="btn secondary" href="sms:{$d['phone1']}">✉️ Send SMS</a>
          PLACEHOLDER_WHATSAPP_BTN
        </div>
      </aside>
    </section>
    <footer>Tap a phone or email to connect. Scan the QR to add the contact instantly.</footer>
  </div>
</body>
</html>
HTML;
}

$done = false;
$links = [];

if($_SERVER['REQUEST_METHOD'] === 'POST'){
  // Gather & sanitize
  $first = trim($_POST['first_name'] ?? '');
  $last  = trim($_POST['last_name'] ?? '');
  $title = trim($_POST['title'] ?? 'Director');
  $org   = trim($_POST['org'] ?? '');

  $phone1 = cleanPhone($_POST['phone1'] ?? '');
  $phone2 = cleanPhone($_POST['phone2'] ?? '');
  $whatsapp = cleanPhone($_POST['whatsapp'] ?? '');
  $email1 = trim($_POST['email1'] ?? '');
  $email2 = trim($_POST['email2'] ?? '');

  $street = trim($_POST['street'] ?? '101, 1st Floor, Todi Building, Mathuradas Mill Compound, Senapati Bapat Marg');
  $city   = trim($_POST['city'] ?? 'Mumbai');
  $region = trim($_POST['region'] ?? 'Maharashtra');
  $postal = trim($_POST['postal'] ?? '400013');
  $country= trim($_POST['country'] ?? 'India');

  $website= trim($_POST['website'] ?? '');
  $logo   = trim($_POST['logo'] ?? '');
  $brand  = trim($_POST['brand'] ?? '#16a34a');
  $slug   = slugify($_POST['slug'] ?? ($first ?: 'contact'));

  // Social media links
  $linkedin = trim($_POST['linkedin'] ?? '');
  $twitter = trim($_POST['twitter'] ?? '');
  $instagram = trim($_POST['instagram'] ?? '');
  $facebook = trim($_POST['facebook'] ?? '');

  if($first === '' || $last === '' || $email1 === ''){
    $err = "First name, last name and primary email are required.";
  } else {
    // Build data
    $data = [
      'first_name'=>$first, 'last_name'=>$last, 'title'=>$title, 'org'=>$org,
      'phone1'=>$phone1, 'phone2'=>$phone2, 'whatsapp'=>$whatsapp, 'email1'=>$email1, 'email2'=>$email2,
      'street'=>$street, 'city'=>$city, 'region'=>$region, 'postal'=>$postal, 'country'=>$country,
      'website'=>$website, 'logo'=>$logo, 'brand'=>$brand,
      'linkedin'=>$linkedin, 'twitter'=>$twitter, 'instagram'=>$instagram, 'facebook'=>$facebook
    ];

    $vcard = buildVCard($data);
    $data['vcard_raw'] = $vcard;

    // Paths
    $baseDir = __DIR__ . DIRECTORY_SEPARATOR . $slug;
    if(!is_dir($baseDir)) mkdir($baseDir, 0755, true);

    $vcfName = h($first . '_' . $last . '.vcf');
    $vcfPath = $baseDir . DIRECTORY_SEPARATOR . $vcfName;
    file_put_contents($vcfPath, $vcard);

    // Html: phones rendered
    $p1html = $phone1 ? '<div class="item"><a href="tel:'.h($phone1).'">📞 <strong>'.h($phone1).'</strong> <span class="badge">mobile</span></a></div>' : '';
    $p2html = $phone2 ? '<div class="item"><a href="tel:'.h($phone2).'">📞 <strong>'.h($phone2).'</strong> <span class="badge">mobile</span></a></div>' : '';
    $whatsapphtml = $whatsapp ? '<div class="item"><a href="https://wa.me/'.str_replace('+', '', h($whatsapp)).'">💬 <strong>'.h($whatsapp).'</strong> <span class="badge">WhatsApp</span></a></div>' : '';
    $data['phone1_display'] = $p1html;
    $data['phone2_display'] = $p2html;
    $data['whatsapp_display'] = $whatsapphtml;

    $data['public_url'] = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']);
    $html = buildHtml($data, $vcfName);
    if($email2){
      $html = str_replace('PLACEHOLDER_EMAIL2', '<div class="item"><a href="mailto:'.h($email2).'">✉️ '.h($email2).'</a></div>', $html);
    } else {
      $html = str_replace('PLACEHOLDER_EMAIL2', '', $html);
    }
    if($whatsapp){
      $whatsappBtn = '<a class="btn secondary" href="https://wa.me/'.str_replace('+', '', h($whatsapp)).'">💬 Send WhatsApp</a>';
      $html = str_replace('PLACEHOLDER_WHATSAPP_BTN', $whatsappBtn, $html);
    } else {
      $html = str_replace('PLACEHOLDER_WHATSAPP_BTN', '', $html);
    }

    // Social media replacements
    $socialIcons = '';
    if($linkedin){
      $socialIcons .= '<a href="'.h($linkedin).'" target="_blank" rel="noopener" class="social-icon linkedin"><i class="fab fa-linkedin-in"></i></a>';
    }
    if($twitter){
      $socialIcons .= '<a href="'.h($twitter).'" target="_blank" rel="noopener" class="social-icon twitter"><i class="fab fa-twitter"></i></a>';
    }
    if($instagram){
      $socialIcons .= '<a href="'.h($instagram).'" target="_blank" rel="noopener" class="social-icon instagram"><i class="fab fa-instagram"></i></a>';
    }
    if($facebook){
      $socialIcons .= '<a href="'.h($facebook).'" target="_blank" rel="noopener" class="social-icon facebook"><i class="fab fa-facebook-f"></i></a>';
    }
    $html = str_replace('PLACEHOLDER_SOCIAL_ICONS', $socialIcons, $html);

    $indexPath = $baseDir . DIRECTORY_SEPARATOR . 'index.html';
    file_put_contents($indexPath, $html);

    $done = true;
    $baseUrl = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST']
             . rtrim(dirname($_SERVER['REQUEST_URI']), '/').'/'.$slug.'/';
    $links = [
      'page' => $baseUrl . 'index.html',
      'vcf'  => $baseUrl . $vcfName
    ];
  }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>VC Card Maker </title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
:root {
  --primary: #16a34a;
  --primary-dark: #0e8d3c;
  --primary-light: #bbf7d0;
  --secondary: #64748b;
  --success: #22c55e;
  --success-bg: #ecfdf5;
  --error: #ef4444;
  --error-bg: #fef2f2;
  --warning: #f59e0b;
  --warning-bg: #fff7ed;
  --background: #f6fbf7;
  --card-bg: #ffffff;
  --border: #e5e7eb;
  --text: #0b1220;
  --text-muted: #64748b;
  --text-light: #9ca3af;
  --shadow: 0 8px 30px rgba(2, 6, 23, 0.08);
  --shadow-hover: 0 12px 40px rgba(2, 6, 23, 0.12);
  --radius: 16px;
  --radius-sm: 12px;
}

* {
  box-sizing: border-box;
}

body {
  font-family: 'Inter', system-ui, -apple-system, 'Segoe UI', Roboto, Arial, sans-serif;
  margin: 0;
  background: linear-gradient(135deg, #f0fdf4 0%, var(--background) 100%);
  color: var(--text);
  line-height: 1.6;
  min-height: 100vh;
}

.container {
  max-width: 1100px;
  margin: 40px auto;
  padding: 20px;
}

.header {
  text-align: center;
  margin-bottom: 40px;
  background: var(--card-bg);
  padding: 40px 20px;
  border-radius: var(--radius);
  box-shadow: var(--shadow);
  border: 1px solid var(--border);
}

.header h1 {
  margin: 0 0 16px 0;
  font-size: clamp(28px, 5vw, 42px);
  font-weight: 800;
  color: var(--primary);
  text-shadow: 0 2px 4px rgba(22, 163, 74, 0.1);
}

.header .subtitle {
  font-size: 18px;
  color: var(--text-muted);
  margin: 0;
  max-width: 600px;
  margin: 0 auto;
}

.card {
  background: var(--card-bg);
  border: 1px solid var(--border);
  border-radius: var(--radius);
  box-shadow: var(--shadow);
  padding: 40px;
  transition: box-shadow 0.3s ease;
}

.card:hover {
  box-shadow: var(--shadow-hover);
}

.form-section {
  margin-bottom: 40px;
}

.form-section:last-child {
  margin-bottom: 0;
}

.section-header {
  display: flex;
  align-items: center;
  gap: 12px;
  margin-bottom: 24px;
  padding-bottom: 16px;
  border-bottom: 2px solid var(--primary-light);
}

.section-header i {
  width: 24px;
  height: 24px;
  background: var(--primary);
  color: white;
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  font-size: 12px;
}

.section-header h3 {
  margin: 0;
  font-size: 20px;
  font-weight: 700;
  color: var(--text);
}

.grid {
  display: grid;
  grid-template-columns: repeat(2, 1fr);
  gap: 24px;
}

.grid-3 {
  display: grid;
  grid-template-columns: repeat(3, 1fr);
  gap: 24px;
}

.grid-single {
  display: grid;
  grid-template-columns: 1fr;
  gap: 24px;
}

@media (max-width: 840px) {
  .grid, .grid-3 {
    grid-template-columns: 1fr;
  }
}

.form-group {
  position: relative;
}

.form-group label {
  display: block;
  font-weight: 600;
  font-size: 14px;
  color: var(--text);
  margin-bottom: 8px;
  transition: color 0.2s ease;
}

.form-group label.required::after {
  content: ' *';
  color: var(--error);
}

.input-wrapper {
  position: relative;
}

.input-wrapper i {
  position: absolute;
  left: 16px;
  top: 50%;
  transform: translateY(-50%);
  color: var(--text-muted);
  font-size: 16px;
  z-index: 1;
}

.form-group input {
  width: 100%;
  padding: 16px 20px;
  border: 2px solid var(--border);
  border-radius: var(--radius-sm);
  font-size: 16px;
  background: var(--card-bg);
  color: var(--text);
  transition: all 0.3s ease;
  outline: none;
}

.form-group input.with-icon {
  padding-left: 48px;
}

.form-group input:focus {
  border-color: var(--primary);
  box-shadow: 0 0 0 4px rgba(22, 163, 74, 0.1);
  transform: translateY(-1px);
}

.form-group input:valid {
  border-color: var(--success);
}

.form-group input.error {
  border-color: var(--error);
  box-shadow: 0 0 0 4px rgba(239, 68, 68, 0.1);
}

.form-group small {
  display: block;
  margin-top: 6px;
  color: var(--text-light);
  font-size: 13px;
}

.form-group .error-message {
  color: var(--error);
  font-size: 13px;
  margin-top: 6px;
  display: none;
}

.form-group.has-error .error-message {
  display: block;
}

.btn {
  display: inline-flex;
  align-items: center;
  gap: 12px;
  background: var(--primary);
  color: white;
  border: 2px solid var(--primary-dark);
  padding: 16px 32px;
  border-radius: var(--radius-sm);
  text-decoration: none;
  font-weight: 700;
  font-size: 16px;
  cursor: pointer;
  transition: all 0.3s ease;
  position: relative;
  overflow: hidden;
}

.btn:hover {
  background: var(--primary-dark);
  transform: translateY(-2px);
  box-shadow: 0 8px 25px rgba(22, 163, 74, 0.3);
}

.btn:active {
  transform: translateY(0);
}

.btn.loading {
  pointer-events: none;
  opacity: 0.8;
}

.btn.loading::before {
  content: '';
  width: 20px;
  height: 20px;
  border: 2px solid rgba(255, 255, 255, 0.3);
  border-top: 2px solid white;
  border-radius: 50%;
  animation: spin 1s linear infinite;
  margin-right: 8px;
}

@keyframes spin {
  0% { transform: rotate(0deg); }
  100% { transform: rotate(360deg); }
}

.alert {
  padding: 20px 24px;
  border-radius: var(--radius-sm);
  margin-bottom: 24px;
  display: flex;
  align-items: flex-start;
  gap: 12px;
  font-weight: 500;
}

.alert i {
  flex-shrink: 0;
  margin-top: 2px;
}

.alert.success {
  background: var(--success-bg);
  border: 1px solid var(--success);
  color: #15803d;
}

.alert.error {
  background: var(--error-bg);
  border: 1px solid var(--error);
  color: #dc2626;
}

.alert.warning {
  background: var(--warning-bg);
  border: 1px solid var(--warning);
  color: #d97706;
}

.alert .alert-title {
  font-weight: 700;
  margin-bottom: 4px;
}

.alert-actions {
  display: flex;
  gap: 12px;
  margin-top: 16px;
  flex-wrap: wrap;
}

.alert-actions .btn {
  padding: 10px 20px;
  font-size: 14px;
}

.form-footer {
  margin-top: 40px;
  padding-top: 32px;
  border-top: 1px solid var(--border);
  text-align: center;
}

.form-footer small {
  color: var(--text-muted);
  font-size: 14px;
  line-height: 1.5;
}

/* Animation for form sections */
.form-section {
  opacity: 0;
  transform: translateY(20px);
  animation: fadeInUp 0.6s ease forwards;
}

.form-section:nth-child(1) { animation-delay: 0.1s; }
.form-section:nth-child(2) { animation-delay: 0.2s; }
.form-section:nth-child(3) { animation-delay: 0.3s; }
.form-section:nth-child(4) { animation-delay: 0.4s; }
.form-section:nth-child(5) { animation-delay: 0.5s; }

@keyframes fadeInUp {
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

/* Responsive improvements */
@media (max-width: 640px) {
  .container {
    padding: 16px;
  }
  
  .card {
    padding: 24px;
  }
  
  .header {
    padding: 32px 20px;
  }
  
  .btn {
    width: 100%;
    justify-content: center;
  }
}

/* Legacy compatibility */
h4 {
  margin: 20px 0 10px 0;
  color: #374151;
  font-weight: 600;
}

.success {
  background: #ecfdf5;
  border: 1px solid #bbf7d0;
  padding: 16px;
  border-radius: 12px;
  margin-bottom: 16px;
}
</style>
</head>
<body>
  <div class="container">
    <div class="header">
      <h1><i class="fas fa-address-card"></i> VC Card Maker</h1>
      <p class="subtitle">Create professional digital business cards with vCard (.vcf) files and branded landing pages for your team</p>
    </div>

    <?php if($done): ?>
      <div class="alert success">
        <i class="fas fa-check-circle"></i>
        <div>
          <div class="alert-title">Success! Your card has been generated.</div>
          <p>Your professional digital business card is ready to share.</p>
          <p style="font-family: monospace; background: rgba(22, 163, 74, 0.1); padding: 8px 12px; border-radius: 6px; color: var(--primary); font-size: 14px;">
            <i class="fas fa-link"></i> Available at: <strong>/vc/<?=h($slug)?>/</strong>
          </p>
          <div class="alert-actions">
            <a class="btn" href="<?=h($links['page'])?>" target="_blank">
              <i class="fas fa-external-link-alt"></i> View Landing Page
            </a>
            <a class="btn" href="<?=h($links['vcf'])?>" target="_blank">
              <i class="fas fa-download"></i> Download vCard
            </a>
          </div>
        </div>
      </div>
    <?php elseif(!empty($err)): ?>
      <div class="alert error">
        <i class="fas fa-exclamation-triangle"></i>
        <div>
          <div class="alert-title">Error occurred</div>
          <p><?=h($err)?></p>
        </div>
      </div>
    <?php endif; ?>

    <form method="post" class="card" id="vcardForm" role="form" aria-label="Create Digital Business Card">
      <!-- Personal Information Section -->
      <div class="form-section">
        <div class="section-header">
          <i class="fas fa-user" aria-hidden="true"></i>
          <h3>Personal Information</h3>
        </div>
        <div class="grid">
          <div class="form-group">
            <label for="first_name" class="required">First Name</label>
            <div class="input-wrapper">
              <i class="fas fa-user" aria-hidden="true"></i>
              <input id="first_name" name="first_name" type="text" class="with-icon" required 
                     value="<?=h($_POST['first_name'] ?? '')?>" 
                     placeholder="Enter first name"
                     aria-describedby="first_name_error"
                     autocomplete="given-name">
            </div>
            <div class="error-message" id="first_name_error" role="alert" aria-live="polite">First name is required</div>
          </div>
          <div class="form-group">
            <label for="last_name" class="required">Last Name</label>
            <div class="input-wrapper">
              <i class="fas fa-user" aria-hidden="true"></i>
              <input id="last_name" name="last_name" type="text" class="with-icon" required 
                     value="<?=h($_POST['last_name'] ?? '')?>" 
                     placeholder="Enter last name"
                     aria-describedby="last_name_error"
                     autocomplete="family-name">
            </div>
            <div class="error-message" id="last_name_error" role="alert" aria-live="polite">Last name is required</div>
          </div>
          <div class="form-group">
            <label for="title">Job Title</label>
            <div class="input-wrapper">
              <i class="fas fa-briefcase" aria-hidden="true"></i>
              <input id="title" name="title" type="text" class="with-icon" 
                     value="<?=h($_POST['title'] ?? 'Director')?>" 
                     placeholder="e.g., Director, Manager"
                     autocomplete="organization-title">
            </div>
          </div>
          <div class="form-group">
            <label for="org">Organization</label>
            <div class="input-wrapper">
              <i class="fas fa-building" aria-hidden="true"></i>
              <input id="org" name="org" type="text" class="with-icon" 
                     value="<?=h($_POST['org'] ?? '')?>" 
                     placeholder="Company name"
                     autocomplete="organization">
            </div>
          </div>
        </div>
      </div>

      <!-- Contact Information Section -->
      <div class="form-section">
        <div class="section-header">
          <i class="fas fa-phone" aria-hidden="true"></i>
          <h3>Contact Information</h3>
        </div>
        <div class="grid">
          <div class="form-group">
            <label for="phone1">Primary Phone</label>
            <div class="input-wrapper">
              <i class="fas fa-phone" aria-hidden="true"></i>
              <input id="phone1" name="phone1" type="tel" class="with-icon" 
                     value="<?=h($_POST['phone1'] ?? '')?>" 
                     placeholder="+91 9876543210"
                     aria-describedby="phone1_help"
                     autocomplete="tel">
            </div>
            <small id="phone1_help">Include country code or we'll add +91</small>
          </div>
          <div class="form-group">
            <label for="phone2">Secondary Phone</label>
            <div class="input-wrapper">
              <i class="fas fa-phone-alt" aria-hidden="true"></i>
              <input id="phone2" name="phone2" type="tel" class="with-icon" 
                     value="<?=h($_POST['phone2'] ?? '')?>" 
                     placeholder="+91 9876543211"
                     aria-describedby="phone2_help"
                     autocomplete="tel">
            </div>
            <small id="phone2_help">Optional secondary number</small>
          </div>
          <div class="form-group">
            <label for="whatsapp">WhatsApp Number</label>
            <div class="input-wrapper">
              <i class="fab fa-whatsapp" aria-hidden="true"></i>
              <input id="whatsapp" name="whatsapp" type="tel" class="with-icon" 
                     value="<?=h($_POST['whatsapp'] ?? '')?>" 
                     placeholder="+91 9876543210"
                     aria-describedby="whatsapp_help"
                     autocomplete="tel">
            </div>
            <small id="whatsapp_help">For WhatsApp quick contact</small>
          </div>
          <div class="form-group">
            <label for="email1" class="required">Primary Email</label>
            <div class="input-wrapper">
              <i class="fas fa-envelope" aria-hidden="true"></i>
              <input id="email1" name="email1" type="email" class="with-icon" required 
                     value="<?=h($_POST['email1'] ?? '')?>" 
                     placeholder="john@company.com"
                     aria-describedby="email1_error"
                     autocomplete="email">
            </div>
            <div class="error-message" id="email1_error" role="alert" aria-live="polite">Valid email address is required</div>
          </div>
          <div class="form-group">
            <label for="email2">Secondary Email</label>
            <div class="input-wrapper">
              <i class="fas fa-envelope-open" aria-hidden="true"></i>
              <input id="email2" name="email2" type="email" class="with-icon" 
                     value="<?=h($_POST['email2'] ?? '')?>" 
                     placeholder="john.doe@company.com"
                     aria-describedby="email2_help"
                     autocomplete="email">
            </div>
            <small id="email2_help">Optional secondary email</small>
          </div>
        </div>
      </div>

      <!-- Address Information Section -->
      <div class="form-section">
        <div class="section-header">
          <i class="fas fa-map-marker-alt" aria-hidden="true"></i>
          <h3>Address Information</h3>
        </div>
        <div class="grid-single">
          <div class="form-group">
            <label for="street">Street Address</label>
            <div class="input-wrapper">
              <i class="fas fa-road" aria-hidden="true"></i>
              <input id="street" name="street" type="text" class="with-icon" 
                     value="<?=h($_POST['street'] ?? '101, 1st Floor, Todi Building, Mathuradas Mill Compound, Senapati Bapat Marg')?>" 
                     placeholder="Building, Street, Area"
                     autocomplete="street-address">
            </div>
          </div>
        </div>
        <div class="grid-3">
          <div class="form-group">
            <label for="city">City</label>
            <div class="input-wrapper">
              <i class="fas fa-city" aria-hidden="true"></i>
              <input id="city" name="city" type="text" class="with-icon" 
                     value="<?=h($_POST['city'] ?? 'Mumbai')?>" 
                     placeholder="Mumbai"
                     autocomplete="address-level2">
            </div>
          </div>
          <div class="form-group">
            <label for="region">State/Region</label>
            <div class="input-wrapper">
              <i class="fas fa-map" aria-hidden="true"></i>
              <input id="region" name="region" type="text" class="with-icon" 
                     value="<?=h($_POST['region'] ?? 'Maharashtra')?>" 
                     placeholder="Maharashtra"
                     autocomplete="address-level1">
            </div>
          </div>
          <div class="form-group">
            <label for="postal">Pincode</label>
            <div class="input-wrapper">
              <i class="fas fa-mail-bulk" aria-hidden="true"></i>
              <input id="postal" name="postal" type="text" class="with-icon" 
                     value="<?=h($_POST['postal'] ?? '400013')?>" 
                     placeholder="400013"
                     autocomplete="postal-code">
            </div>
          </div>
        </div>
        <div class="grid">
          <div class="form-group">
            <label for="country">Country</label>
            <div class="input-wrapper">
              <i class="fas fa-globe" aria-hidden="true"></i>
              <input id="country" name="country" type="text" class="with-icon" 
                     value="<?=h($_POST['country'] ?? 'India')?>" 
                     placeholder="India"
                     autocomplete="country-name">
            </div>
          </div>
        </div>
      </div>

      <!-- Online Presence Section -->
      <div class="form-section">
        <div class="section-header">
          <i class="fas fa-globe-americas" aria-hidden="true"></i>
          <h3>Online Presence</h3>
        </div>
        <div class="grid">
          <div class="form-group">
            <label for="website">Website URL</label>
            <div class="input-wrapper">
              <i class="fas fa-link" aria-hidden="true"></i>
              <input id="website" name="website" type="url" class="with-icon" 
                     value="<?=h($_POST['website'] ?? '')?>" 
                     placeholder="https://yourwebsite.com"
                     autocomplete="url">
            </div>
          </div>
          <div class="form-group">
            <label for="logo">Logo URL</label>
            <div class="input-wrapper">
              <i class="fas fa-image" aria-hidden="true"></i>
              <input id="logo" name="logo" type="url" class="with-icon" 
                     value="<?=h($_POST['logo'] ?? '')?>" 
                     placeholder="https://example.com/logo.png"
                     aria-describedby="logo_help">
            </div>
            <small id="logo_help">Company or personal logo image URL</small>
          </div>
        </div>
        
        <h4 style="margin-top: 32px; margin-bottom: 16px; color: var(--text-muted);">
          <i class="fas fa-share-alt" style="margin-right: 8px;" aria-hidden="true"></i>Social Media Profiles
        </h4>
        <div class="grid">
          <div class="form-group">
            <label for="linkedin">LinkedIn Profile</label>
            <div class="input-wrapper">
              <i class="fab fa-linkedin" aria-hidden="true"></i>
              <input id="linkedin" name="linkedin" type="url" class="with-icon" 
                     value="<?=h($_POST['linkedin'] ?? '')?>" 
                     placeholder="https://linkedin.com/in/username"
                     autocomplete="url">
            </div>
          </div>
          <div class="form-group">
            <label for="twitter">Twitter/X Profile</label>
            <div class="input-wrapper">
              <i class="fab fa-twitter" aria-hidden="true"></i>
              <input id="twitter" name="twitter" type="url" class="with-icon" 
                     value="<?=h($_POST['twitter'] ?? '')?>" 
                     placeholder="https://twitter.com/username"
                     autocomplete="url">
            </div>
          </div>
          <div class="form-group">
            <label for="instagram">Instagram Profile</label>
            <div class="input-wrapper">
              <i class="fab fa-instagram" aria-hidden="true"></i>
              <input id="instagram" name="instagram" type="url" class="with-icon" 
                     value="<?=h($_POST['instagram'] ?? '')?>" 
                     placeholder="https://instagram.com/username"
                     autocomplete="url">
            </div>
          </div>
          <div class="form-group">
            <label for="facebook">Facebook Profile</label>
            <div class="input-wrapper">
              <i class="fab fa-facebook" aria-hidden="true"></i>
              <input id="facebook" name="facebook" type="url" class="with-icon" 
                     value="<?=h($_POST['facebook'] ?? '')?>" 
                     placeholder="https://facebook.com/username"
                     autocomplete="url">
            </div>
          </div>
        </div>
      </div>

      <!-- Customization Section -->
      <div class="form-section">
        <div class="section-header">
          <i class="fas fa-palette" aria-hidden="true"></i>
          <h3>Customization</h3>
        </div>
        <div class="grid">
          <div class="form-group">
            <label for="brand">Brand Color</label>
            <div class="input-wrapper">
              <i class="fas fa-paint-brush" aria-hidden="true"></i>
              <input id="brand" name="brand" type="color" class="with-icon" 
                     value="<?=h($_POST['brand'] ?? '#16a34a')?>" 
                     style="padding-left: 48px;"
                     aria-describedby="brand_help">
            </div>
            <small id="brand_help">Choose your brand's primary color</small>
          </div>
          <div class="form-group">
            <label for="slug">URL Slug</label>
            <div class="input-wrapper">
              <i class="fas fa-link" aria-hidden="true"></i>
              <input id="slug" name="slug" type="text" class="with-icon" 
                     value="<?=h($_POST['slug'] ?? '')?>" 
                     placeholder="john-doe"
                     aria-describedby="slug_help"
                     pattern="[a-z0-9\-]+"
                     title="Only lowercase letters, numbers, and hyphens allowed">
            </div>
            <small id="slug_help">Creates /vc/&lt;slug&gt;/ URL path</small>
          </div>
        </div>
      </div>

      <div class="form-footer">
        <button class="btn" type="submit" id="submitBtn" aria-describedby="submit_help">
          <i class="fas fa-magic" aria-hidden="true"></i> Generate Professional Card
        </button>
        <br><br>
        <small id="submit_help">
          <i class="fas fa-info-circle" aria-hidden="true"></i>
          Files are created in a folder named by the slug next to this PHP file.
          <br>Your digital business card will include a QR code and be optimized for mobile sharing.
        </small>
      </div>
    </form>
  </div>

  <script>
    // Form enhancement script
    document.addEventListener('DOMContentLoaded', function() {
      const form = document.getElementById('vcardForm');
      const submitBtn = document.getElementById('submitBtn');
      
      // Add loading state on form submission
      form.addEventListener('submit', function() {
        submitBtn.classList.add('loading');
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generating Card...';
        submitBtn.disabled = true;
      });
      
      // Basic client-side validation
      const requiredFields = form.querySelectorAll('input[required]');
      
      requiredFields.forEach(field => {
        field.addEventListener('blur', validateField);
        field.addEventListener('input', clearError);
      });
      
      function validateField(e) {
        const field = e.target;
        const formGroup = field.closest('.form-group');
        
        if (!field.value.trim()) {
          formGroup.classList.add('has-error');
          field.classList.add('error');
        } else if (field.type === 'email' && !isValidEmail(field.value)) {
          formGroup.classList.add('has-error');
          field.classList.add('error');
        } else {
          formGroup.classList.remove('has-error');
          field.classList.remove('error');
        }
      }
      
      function clearError(e) {
        const field = e.target;
        const formGroup = field.closest('.form-group');
        formGroup.classList.remove('has-error');
        field.classList.remove('error');
      }
      
      function isValidEmail(email) {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
      }
      
      // Auto-generate slug from name
      const firstNameField = form.querySelector('input[name="first_name"]');
      const lastNameField = form.querySelector('input[name="last_name"]');
      const slugField = form.querySelector('input[name="slug"]');
      
      function updateSlug() {
        if (!slugField.value && firstNameField.value && lastNameField.value) {
          const slug = (firstNameField.value + '-' + lastNameField.value)
            .toLowerCase()
            .replace(/[^a-z0-9\-]/g, '-')
            .replace(/-+/g, '-')
            .replace(/^-|-$/g, '');
          slugField.value = slug;
        }
      }
      
      firstNameField.addEventListener('input', updateSlug);
      lastNameField.addEventListener('input', updateSlug);
    });
  </script>
</body>
</html>
