<?php
/**
 * Secure Two Factor Authentication - Admin Interface
 */

if (!defined('ABSPATH')) {
    exit;
}

class S2FA_Admin {
    
    public function __construct() {
        add_action('show_user_profile', array($this, 'show_user_profile_fields'));
        add_action('edit_user_profile', array($this, 'show_user_profile_fields'));
        add_action('personal_options_update', array($this, 'save_user_profile_fields'));
        add_action('edit_user_profile_update', array($this, 'save_user_profile_fields'));
        add_action('wp_ajax_s2fa_setup', array($this, 'handle_ajax_setup'));
        add_action('wp_ajax_s2fa_verify', array($this, 'handle_ajax_verify'));
        add_action('wp_ajax_s2fa_disable', array($this, 'handle_ajax_disable'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Show 2FA fields on user profile page
     */
    public function show_user_profile_fields($user) {
        $is_enabled = S2FA_User::is_2fa_enabled($user->ID);
        $can_edit = current_user_can('edit_user', $user->ID);
        
        if (!$can_edit) {
            return;
        }
        
        ?>
        <h3><?php _e('Two-Factor Authentication', 'secure-two-factor-auth'); ?></h3>
        <table class="form-table">
            <tr>
                <th><label><?php _e('Status', 'secure-two-factor-auth'); ?></label></th>
                <td>
                    <div id="s2fa-status-container">
                        <?php if ($is_enabled): ?>
                            <span class="s2fa-status enabled">
                                <span class="dashicons dashicons-yes-alt"></span>
                                <?php _e('Two-Factor Authentication is enabled', 'secure-two-factor-auth'); ?>
                            </span>
                            <p class="description">
                                <?php _e('Your account is protected with two-factor authentication.', 'secure-two-factor-auth'); ?>
                            </p>
                            <p>
                                <button type="button" id="s2fa-disable-btn" class="button button-secondary">
                                    <?php _e('Disable Two-Factor Authentication', 'secure-two-factor-auth'); ?>
                                </button>
                                <button type="button" id="s2fa-regenerate-codes-btn" class="button button-secondary">
                                    <?php _e('Generate New Backup Codes', 'secure-two-factor-auth'); ?>
                                </button>
                            </p>
                        <?php else: ?>
                            <span class="s2fa-status disabled">
                                <span class="dashicons dashicons-warning"></span>
                                <?php _e('Two-Factor Authentication is disabled', 'secure-two-factor-auth'); ?>
                            </span>
                            <p class="description">
                                <?php _e('Enable two-factor authentication to add an extra layer of security to your account.', 'secure-two-factor-auth'); ?>
                            </p>
                            <p>
                                <button type="button" id="s2fa-setup-btn" class="button button-primary">
                                    <?php _e('Enable Two-Factor Authentication', 'secure-two-factor-auth'); ?>
                                </button>
                            </p>
                        <?php endif; ?>
                    </div>
                </td>
            </tr>
        </table>
        
        <!-- Setup Modal -->
        <div id="s2fa-setup-modal" class="s2fa-modal" style="display: none;">
            <div class="s2fa-modal-content">
                <div class="s2fa-modal-header">
                    <h2><?php _e('Set up Two-Factor Authentication', 'secure-two-factor-auth'); ?></h2>
                    <span class="s2fa-modal-close">&times;</span>
                </div>
                <div class="s2fa-modal-body">
                    <div id="s2fa-setup-step-1" class="s2fa-setup-step">
                        <h3><?php _e('Step 1: Install an Authenticator App', 'secure-two-factor-auth'); ?></h3>
                        <p><?php _e('Install one of these authenticator apps on your mobile device:', 'secure-two-factor-auth'); ?></p>
                        <ul class="s2fa-app-list">
                            <li><strong>Google Authenticator</strong> - iOS / Android</li>
                            <li><strong>Microsoft Authenticator</strong> - iOS / Android</li>
                            <li><strong>Authy</strong> - iOS / Android / Desktop</li>
                        </ul>
                        <p>
                            <button type="button" id="s2fa-next-step" class="button button-primary">
                                <?php _e('I have installed an app', 'secure-two-factor-auth'); ?>
                            </button>
                        </p>
                    </div>
                    
                    <div id="s2fa-setup-step-2" class="s2fa-setup-step" style="display: none;">
                        <h3><?php _e('Step 2: Scan QR Code or Enter Secret', 'secure-two-factor-auth'); ?></h3>
                        <div id="s2fa-qr-container" class="s2fa-qr-container">
                            <!-- QR code will be loaded here -->
                        </div>
                        <div class="s2fa-manual-entry">
                            <p><strong><?php _e('Can\'t scan? Enter this code manually:', 'secure-two-factor-auth'); ?></strong></p>
                            <code id="s2fa-secret-display"></code>
                        </div>
                        <p>
                            <button type="button" id="s2fa-verify-step" class="button button-primary">
                                <?php _e('I have added the account', 'secure-two-factor-auth'); ?>
                            </button>
                        </p>
                    </div>
                    
                    <div id="s2fa-setup-step-3" class="s2fa-setup-step" style="display: none;">
                        <h3><?php _e('Step 3: Verify Setup', 'secure-two-factor-auth'); ?></h3>
                        <p><?php _e('Enter the 6-digit code from your authenticator app to verify the setup:', 'secure-two-factor-auth'); ?></p>
                        <p>
                            <input type="text" id="s2fa-verify-code" class="regular-text" placeholder="000000" maxlength="6" style="text-align: center; font-size: 18px; letter-spacing: 2px;" />
                        </p>
                        <p>
                            <button type="button" id="s2fa-complete-setup" class="button button-primary">
                                <?php _e('Verify and Enable', 'secure-two-factor-auth'); ?>
                            </button>
                        </p>
                    </div>
                    
                    <div id="s2fa-setup-step-4" class="s2fa-setup-step" style="display: none;">
                        <h3><?php _e('Setup Complete!', 'secure-two-factor-auth'); ?></h3>
                        <p><?php _e('Two-factor authentication has been enabled for your account.', 'secure-two-factor-auth'); ?></p>
                        <div id="s2fa-backup-codes-container">
                            <h4><?php _e('Important: Save Your Backup Codes', 'secure-two-factor-auth'); ?></h4>
                            <p><?php _e('These backup codes can be used if you lose access to your authenticator app. Save them in a safe place:', 'secure-two-factor-auth'); ?></p>
                            <div id="s2fa-backup-codes-list"></div>
                            <p class="description">
                                <?php _e('Each code can only be used once. You can generate new codes anytime from your profile.', 'secure-two-factor-auth'); ?>
                            </p>
                        </div>
                        <p>
                            <button type="button" id="s2fa-finish-setup" class="button button-primary">
                                <?php _e('Finish', 'secure-two-factor-auth'); ?>
                            </button>
                        </p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Disable Modal -->
        <div id="s2fa-disable-modal" class="s2fa-modal" style="display: none;">
            <div class="s2fa-modal-content">
                <div class="s2fa-modal-header">
                    <h2><?php _e('Disable Two-Factor Authentication', 'secure-two-factor-auth'); ?></h2>
                    <span class="s2fa-modal-close">&times;</span>
                </div>
                <div class="s2fa-modal-body">
                    <p><?php _e('Are you sure you want to disable two-factor authentication? This will make your account less secure.', 'secure-two-factor-auth'); ?></p>
                    <p>
                        <button type="button" id="s2fa-confirm-disable" class="button button-secondary">
                            <?php _e('Yes, Disable 2FA', 'secure-two-factor-auth'); ?>
                        </button>
                        <button type="button" class="button button-primary s2fa-modal-close">
                            <?php _e('Cancel', 'secure-two-factor-auth'); ?>
                        </button>
                    </p>
                </div>
            </div>
        </div>
        
        <input type="hidden" id="s2fa-user-id" value="<?php echo esc_attr($user->ID); ?>" />
        <input type="hidden" id="s2fa-nonce" value="<?php echo wp_create_nonce('s2fa_admin_nonce'); ?>" />
        <?php
    }
    
    /**
     * Save user profile fields (placeholder for now)
     */
    public function save_user_profile_fields($user_id) {
        // 2FA settings are handled via AJAX
        // This is kept for compatibility
    }
    
    /**
     * Handle AJAX setup request
     */
    public function handle_ajax_setup() {
        try {
            if (!wp_verify_nonce($_POST['nonce'], 's2fa_admin_nonce')) {
                wp_send_json_error('Security check failed');
                return;
            }
            
            $user_id = intval($_POST['user_id']);
            
            if (!current_user_can('edit_user', $user_id)) {
                wp_send_json_error('Permission denied');
                return;
            }
            
            // Generate secret
            $secret = S2FA_User::generate_secret($user_id);
            if (!$secret) {
                wp_send_json_error('Failed to generate secret');
                return;
            }
            
            $setup_data = S2FA_User::get_setup_instructions($user_id);
            if (!$setup_data) {
                wp_send_json_error('Failed to get setup instructions');
                return;
            }
            
            wp_send_json_success($setup_data);
        } catch (Exception $e) {
            wp_send_json_error('Setup failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Handle AJAX verification and enable
     */
    public function handle_ajax_verify() {
        if (!wp_verify_nonce($_POST['nonce'], 's2fa_admin_nonce')) {
            wp_die('Security check failed');
        }
        
        $user_id = intval($_POST['user_id']);
        $code = sanitize_text_field($_POST['code']);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_die('Permission denied');
        }
        
        $secret = S2FA_User::get_user_secret($user_id);
        
        if (S2FA_TOTP::verify_code($code, $secret)) {
            S2FA_User::enable_2fa($user_id);
            $backup_codes = S2FA_User::generate_backup_codes($user_id);
            
            wp_send_json_success(array(
                'backup_codes' => $backup_codes
            ));
        } else {
            wp_send_json_error('Invalid verification code');
        }
    }
    
    /**
     * Handle AJAX disable request
     */
    public function handle_ajax_disable() {
        if (!wp_verify_nonce($_POST['nonce'], 's2fa_admin_nonce')) {
            wp_die('Security check failed');
        }
        
        $user_id = intval($_POST['user_id']);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_die('Permission denied');
        }
        
        S2FA_User::disable_2fa($user_id);
        
        wp_send_json_success('Two-factor authentication disabled');
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook_suffix) {
        if ($hook_suffix !== 'profile.php' && $hook_suffix !== 'user-edit.php') {
            return;
        }
        
        wp_enqueue_script('jquery');
        
        wp_add_inline_style('admin-bar', $this->get_admin_styles());
        wp_add_inline_script('jquery', $this->get_admin_javascript());
    }
    
    /**
     * Get admin styles
     */
    private function get_admin_styles() {
        return '
            .s2fa-status {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 8px 12px;
                border-radius: 4px;
                font-weight: 500;
            }
            
            .s2fa-status.enabled {
                background-color: #d1e7dd;
                color: #0a3622;
                border: 1px solid #a3cfbb;
            }
            
            .s2fa-status.disabled {
                background-color: #f8d7da;
                color: #721c24;
                border: 1px solid #f1aeb5;
            }
            
            .s2fa-modal {
                position: fixed;
                z-index: 100000;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                overflow: auto;
                background-color: rgba(0,0,0,0.5);
            }
            
            .s2fa-modal-content {
                background-color: #fefefe;
                margin: 5% auto;
                padding: 0;
                border: 1px solid #ddd;
                border-radius: 4px;
                width: 90%;
                max-width: 600px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            }
            
            .s2fa-modal-header {
                padding: 16px 20px;
                background-color: #f9f9f9;
                border-bottom: 1px solid #ddd;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            
            .s2fa-modal-header h2 {
                margin: 0;
                font-size: 20px;
            }
            
            .s2fa-modal-close {
                color: #999;
                float: right;
                font-size: 28px;
                font-weight: bold;
                cursor: pointer;
                line-height: 1;
            }
            
            .s2fa-modal-close:hover {
                color: #333;
            }
            
            .s2fa-modal-body {
                padding: 20px;
            }
            
            .s2fa-app-list {
                list-style: none;
                padding: 0;
            }
            
            .s2fa-app-list li {
                padding: 8px 0;
                border-bottom: 1px solid #eee;
            }
            
            .s2fa-qr-container {
                text-align: center;
                margin: 20px 0;
            }
            
            .s2fa-qr-container img {
                border: 1px solid #ddd;
                padding: 10px;
                background: white;
            }
            
            .s2fa-manual-entry {
                background: #f9f9f9;
                padding: 16px;
                border-radius: 4px;
                margin: 16px 0;
                text-align: center;
            }
            
            .s2fa-manual-entry code {
                font-size: 16px;
                font-weight: bold;
                letter-spacing: 2px;
                color: #333;
                background: white;
                padding: 8px 16px;
                border-radius: 4px;
                border: 1px solid #ddd;
            }
            
            #s2fa-backup-codes-list {
                background: #f9f9f9;
                padding: 16px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 14px;
                margin: 16px 0;
                max-height: 200px;
                overflow-y: auto;
            }
            
            .s2fa-backup-code {
                display: block;
                padding: 4px 0;
                font-weight: bold;
            }
        ';
    }
    
    /**
     * Get admin JavaScript
     */
    private function get_admin_javascript() {
        return '
            jQuery(document).ready(function($) {
                var modal = null;
                var currentStep = 1;
                
                // Setup button click
                $("#s2fa-setup-btn").click(function() {
                    showModal("#s2fa-setup-modal");
                    showStep(1);
                });
                
                // Disable button click
                $("#s2fa-disable-btn").click(function() {
                    showModal("#s2fa-disable-modal");
                });
                
                // Modal navigation
                $("#s2fa-next-step").click(function() {
                    setupTwoFactor();
                });
                
                $("#s2fa-verify-step").click(function() {
                    showStep(3);
                });
                
                $("#s2fa-complete-setup").click(function() {
                    verifyAndEnable();
                });
                
                $("#s2fa-finish-setup").click(function() {
                    hideModal();
                    location.reload();
                });
                
                $("#s2fa-confirm-disable").click(function() {
                    disableTwoFactor();
                });
                
                // Close modal
                $(".s2fa-modal-close").click(function() {
                    hideModal();
                });
                
                // Auto-submit verification code
                $("#s2fa-verify-code").on("input", function() {
                    if (this.value.length === 6) {
                        setTimeout(function() {
                            $("#s2fa-complete-setup").click();
                        }, 500);
                    }
                });
                
                function showModal(selector) {
                    modal = $(selector);
                    modal.show();
                }
                
                function hideModal() {
                    if (modal) {
                        modal.hide();
                        modal = null;
                    }
                }
                
                function showStep(step) {
                    $(".s2fa-setup-step").hide();
                    $("#s2fa-setup-step-" + step).show();
                    currentStep = step;
                }
                
                function setupTwoFactor() {
                    $.post(ajaxurl, {
                        action: "s2fa_setup",
                        user_id: $("#s2fa-user-id").val(),
                        nonce: $("#s2fa-nonce").val()
                    }, function(response) {
                        if (response.success) {
                            // Create QR code with error handling
                            var qrImg = new Image();
                            qrImg.onload = function() {
                                $("#s2fa-qr-container").html("<img src=\"" + response.data.qr_url + "\" alt=\"QR Code\" style=\"border: 1px solid #ddd; padding: 10px; background: white;\" />");
                            };
                            qrImg.onerror = function() {
                                $("#s2fa-qr-container").html("<div style=\"padding: 20px; background: #f9f9f9; border: 1px solid #ddd; text-align: center; color: #666;\"><p><strong>QR Code could not be loaded</strong></p><p>Please use manual entry below</p></div>");
                            };
                            qrImg.src = response.data.qr_url;
                            
                            $("#s2fa-secret-display").text(response.data.manual_entry);
                            showStep(2);
                        } else {
                            alert("Error: " + response.data);
                        }
                    }).fail(function() {
                        alert("Connection error. Please try again.");
                    });
                }
                
                function verifyAndEnable() {
                    var code = $("#s2fa-verify-code").val();
                    
                    $.post(ajaxurl, {
                        action: "s2fa_verify",
                        user_id: $("#s2fa-user-id").val(),
                        code: code,
                        nonce: $("#s2fa-nonce").val()
                    }, function(response) {
                        if (response.success) {
                            var codes = response.data.backup_codes;
                            var html = "";
                            for (var i = 0; i < codes.length; i++) {
                                html += "<span class=\"s2fa-backup-code\">" + codes[i] + "</span>";
                            }
                            $("#s2fa-backup-codes-list").html(html);
                            showStep(4);
                        } else {
                            alert("Error: " + response.data);
                        }
                    });
                }
                
                function disableTwoFactor() {
                    $.post(ajaxurl, {
                        action: "s2fa_disable",
                        user_id: $("#s2fa-user-id").val(),
                        nonce: $("#s2fa-nonce").val()
                    }, function(response) {
                        if (response.success) {
                            hideModal();
                            location.reload();
                        } else {
                            alert("Error: " + response.data);
                        }
                    });
                }
            });
        ';
    }
}