<?php
/**
 * Secure Two Factor Authentication - Login Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class S2FA_Login {
    
    public function __construct() {
        add_filter('authenticate', array($this, 'authenticate_user_credentials'), 30, 3);
        add_action('init', array($this, 'handle_2fa_verification'));
        add_action('login_enqueue_scripts', array($this, 'enqueue_login_scripts'));
        add_action('wp_login', array($this, 'handle_2fa_login'), 10, 2);
        add_action('wp_loaded', array($this, 'show_2fa_page'));
    }
    
    /**
     * Authenticate user credentials first, then check 2FA
     */
    public function authenticate_user_credentials($user, $username, $password) {
        // Skip if already an error or empty credentials
        if (is_wp_error($user) || empty($username) || empty($password)) {
            return $user;
        }
        
        // Skip if user doesn't exist
        if (!($user instanceof WP_User)) {
            return $user;
        }
        
        // Check if 2FA is enabled for this user
        if (!S2FA_User::is_2fa_enabled($user->ID)) {
            return $user;
        }
        
        // Store user info in WordPress options instead of transient for more reliability
        $token = substr(md5(uniqid(mt_rand(), true)), 0, 16); // Shorter token
        $option_key = 's2fa_' . $token;
        
        $user_data = array(
            'user_id' => $user->ID,
            'user_login' => $user->user_login,
            'timestamp' => time(),
            'expires' => time() + 1800, // 30 minutes from now (increased from 15 minutes)
            'redirect_to' => isset($_POST['redirect_to']) ? $_POST['redirect_to'] : admin_url()
        );
        
        // Use WordPress option instead of transient
        $result = update_option($option_key, $user_data, false); // Don't autoload
        
        // Add debug logging
        error_log("S2FA: Created option $option_key with data: " . print_r($user_data, true));
        error_log("S2FA: Option creation result: " . ($result ? 'SUCCESS' : 'FAILED'));
        
        // Double-check that option was saved correctly
        $verify_data = get_option($option_key);
        if (!$verify_data) {
            error_log("S2FA: CRITICAL - Option $option_key was not saved properly, trying again...");
            // Try saving again with a different approach
            update_option($option_key, $user_data, true); // Try with autoload
            $verify_data = get_option($option_key);
            if (!$verify_data) {
                error_log("S2FA: CRITICAL - Option $option_key still not saved, login will fail");
                wp_die('2FA session initialization failed. Please contact an administrator.');
            }
        }
        
        // Redirect to 2FA verification page with token
        wp_redirect(wp_login_url() . '?action=2fa_verify&token=' . $token);
        exit;
    }
    
    /**
     * Handle 2FA verification page display
     */
    public function show_2fa_page() {
        if (isset($_GET['action']) && $_GET['action'] === '2fa_verify') {
            // Check if user is already logged in - if so, redirect to admin
            if (is_user_logged_in()) {
                error_log("S2FA: User already logged in, redirecting to admin instead of showing 2FA page");
                wp_redirect(admin_url());
                exit;
            }
            $this->display_2fa_verification_page();
            exit;
        }
    }
    
    /**
     * Handle 2FA verification processing
     */
    public function handle_2fa_verification() {
        if (!isset($_POST['s2fa_verify_submit'])) {
            return;
        }
        
        // Get token from POST data
        $token = isset($_POST['s2fa_token']) ? sanitize_text_field($_POST['s2fa_token']) : '';
        if (empty($token)) {
            wp_redirect(wp_login_url() . '?s2fa_error=invalid_token');
            exit;
        }
        
        // Get user data from option
        $option_key = 's2fa_' . $token;
        $user_data = get_option($option_key);
        if (!$user_data || !isset($user_data['expires'])) {
            wp_redirect(wp_login_url() . '?s2fa_error=session_expired');
            exit;
        }

        // Check if expired
        $current_time = time();
        error_log("S2FA: Current time: " . $current_time . ", Expires: " . $user_data['expires'] . ", Difference: " . ($user_data['expires'] - $current_time) . " seconds");

        if ($current_time > $user_data['expires']) {
            error_log("S2FA: Session expired - Current: " . $current_time . ", Expires: " . $user_data['expires']);
            delete_option($option_key);
            wp_redirect(wp_login_url() . '?s2fa_error=session_expired');
            exit;
        }

        // Refresh session if less than 10 minutes remaining during verification
        $time_remaining = $user_data['expires'] - $current_time;
        if ($time_remaining < 600) { // Less than 10 minutes
            $user_data['expires'] = $current_time + 1800; // Extend by 30 minutes
            update_option($option_key, $user_data, false);
            error_log("S2FA: Session refreshed during verification - new expiry: " . $user_data['expires']);
        }
        
        $user_id = $user_data['user_id'];
        $user = get_user_by('id', $user_id);
        
        if (!$user) {
            delete_option($option_key);
            wp_redirect(wp_login_url() . '?s2fa_error=user_not_found');
            exit;
        }
        
        $error = '';
        $verified = false;
        
        // Verify TOTP code
        if (isset($_POST['s2fa_code']) && !empty($_POST['s2fa_code'])) {
            $code = sanitize_text_field($_POST['s2fa_code']);
            error_log("S2FA: Attempting to verify TOTP code: " . $code . " for user " . $user_id);
            if ($this->verify_totp_code($user_id, $code)) {
                $verified = true;
                error_log("S2FA: TOTP code verification SUCCESS for user " . $user_id);
            } else {
                $error = 'Invalid authenticator code. Please try again.';
                error_log("S2FA: TOTP code verification FAILED for user " . $user_id);
            }
        }
        
        // Verify backup code
        if (!$verified && isset($_POST['s2fa_backup_code']) && !empty($_POST['s2fa_backup_code'])) {
            $backup_code = sanitize_text_field($_POST['s2fa_backup_code']);
            error_log("S2FA: Attempting to verify backup code for user " . $user_id);
            if ($this->verify_backup_code($user_id, $backup_code)) {
                $verified = true;
                error_log("S2FA: Backup code verification SUCCESS for user " . $user_id);
            } else {
                $error = 'Invalid backup code. Please try again.';
                error_log("S2FA: Backup code verification FAILED for user " . $user_id);
            }
        }
        
        if ($verified) {
            // Complete login - try a different approach
            $redirect_to = $user_data['redirect_to'];
            
            error_log("S2FA: Login verified, redirecting to: " . $redirect_to);
            
            // Make sure we remove any 2FA related query parameters from redirect URL
            $redirect_to = remove_query_arg(array('action', 'token', 's2fa_error'), $redirect_to);
            
            error_log("S2FA: About to redirect to (cleaned): " . $redirect_to);
            
            // Clean up the option BEFORE login to prevent any conflicts
            delete_option($option_key);
            
            // Instead of manually setting cookies, let's use WordPress login process
            // Sign the user in using WordPress's secure method
            wp_clear_auth_cookie(); // Clear any existing auth cookies first
            
            // Set auth cookie with proper parameters
            wp_set_auth_cookie($user_id, false, is_ssl());
            wp_set_current_user($user_id);
            
            // Verify the user is properly logged in
            $logged_in_check = is_user_logged_in();
            error_log("S2FA: After setting auth cookie, is_user_logged_in(): " . ($logged_in_check ? 'TRUE' : 'FALSE'));
            error_log("S2FA: Current user ID: " . get_current_user_id());
            
            // Trigger the wp_login action
            do_action('wp_login', $user->user_login, $user);
            
            // Add a small delay to ensure cookies are set
            usleep(100000); // 0.1 second delay
            
            // Force headers to prevent caching
            if (!headers_sent()) {
                header('Cache-Control: no-cache, no-store, must-revalidate');
                header('Pragma: no-cache');
                header('Expires: 0');
            }
            
            // Try immediate redirect with proper WordPress redirect
            if (!headers_sent()) {
                error_log("S2FA: Performing wp_safe_redirect to: " . $redirect_to);
                // Instead of immediate redirect, show thank you page first
                $this->show_thank_you_page($redirect_to);
                exit;
            } else {
                error_log("S2FA: Headers already sent, using JavaScript redirect with thank you page");
                $this->show_thank_you_page($redirect_to);
                exit;
            }
        } else {
            if (empty($error)) {
                $error = 'Please enter an authenticator code or backup code.';
            }
            wp_redirect(wp_login_url() . '?action=2fa_verify&token=' . $token . '&s2fa_error=' . urlencode($error));
            exit;
        }
    }
    
    /**
     * Display 2FA verification page
     */
    private function display_2fa_verification_page() {
        // Get token from URL
        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
        error_log("S2FA: Display page - received token: " . $token);
        
        if (empty($token)) {
            error_log("S2FA: No token provided in URL");
            // If user is already logged in, redirect to admin instead of showing error
            if (is_user_logged_in()) {
                error_log("S2FA: No token but user logged in, redirecting to admin");
                wp_redirect(admin_url());
                exit;
            }
            wp_redirect(wp_login_url() . '?s2fa_error=invalid_token');
            exit;
        }
        
        // Get user data from option
        $option_key = 's2fa_' . $token;
        $user_data = get_option($option_key);
        
        error_log("S2FA: Looking for option: " . $option_key);
        error_log("S2FA: Option data found: " . ($user_data ? 'YES' : 'NO'));
        
        if (!$user_data || !isset($user_data['expires'])) {
            error_log("S2FA: Option expired or not found");
            // Check if user is already logged in - if so, redirect instead of error
            if (is_user_logged_in()) {
                error_log("S2FA: Session expired but user already logged in, redirecting to admin");
                wp_redirect(admin_url());
                exit;
            }
            wp_redirect(wp_login_url() . '?s2fa_error=session_expired');
            exit;
        }

        // Check if expired
        $current_time = time();
        if ($current_time > $user_data['expires']) {
            error_log("S2FA: Token expired - Current: " . $current_time . ", Expires: " . $user_data['expires']);
            delete_option($option_key); // Clean up
            // Check if user is already logged in - if so, redirect instead of error
            if (is_user_logged_in()) {
                error_log("S2FA: Token expired but user already logged in, redirecting to admin");
                wp_redirect(admin_url());
                exit;
            }
            wp_redirect(wp_login_url() . '?s2fa_error=session_expired');
            exit;
        }

        // Refresh session if less than 5 minutes remaining (extend by 30 minutes from current time)
        $time_remaining = $user_data['expires'] - $current_time;
        if ($time_remaining < 300) { // Less than 5 minutes
            $user_data['expires'] = $current_time + 1800; // Extend by 30 minutes
            update_option($option_key, $user_data, false);
            error_log("S2FA: Session refreshed - new expiry: " . $user_data['expires']);
        }
        
        $user_login = $user_data['user_login'];
        $error_message = isset($_GET['s2fa_error']) ? sanitize_text_field($_GET['s2fa_error']) : '';
        
        error_log("S2FA: Displaying 2FA page for user: " . $user_login);
        
        // Get WordPress login page styling
        wp_enqueue_style('login');
        wp_enqueue_script('jquery');
        
        ?><!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta http-equiv="Content-Type" content="<?php bloginfo('html_type'); ?>; charset=<?php echo get_option('blog_charset'); ?>" />
            <title><?php echo get_bloginfo('name', 'display'); ?> &rsaquo; Two-Factor Authentication</title>
            <?php wp_head(); ?>
            <style>
                body.login {
                    background: #f1f1f1;
                }
                .login #login {
                    width: 400px;
                    padding: 5% 0 0;
                    margin: auto;
                }
                .s2fa-container {
                    background: #fff;
                    padding: 24px;
                    border-radius: 4px;
                    box-shadow: 0 1px 3px rgba(0,0,0,.13);
                }
                .s2fa-header {
                    text-align: center;
                    margin-bottom: 24px;
                }
                .s2fa-header h1 {
                    font-size: 24px;
                    margin: 0 0 8px 0;
                    font-weight: 400;
                    color: #333;
                }
                .s2fa-user-info {
                    background: #f9f9f9;
                    padding: 16px;
                    border-radius: 4px;
                    margin-bottom: 24px;
                    text-align: center;
                }
                .s2fa-form-section {
                    margin-bottom: 20px;
                }
                .s2fa-form-section label {
                    display: block;
                    margin-bottom: 8px;
                    font-weight: 600;
                    color: #333;
                }
                .s2fa-input {
                    width: 100%;
                    padding: 12px;
                    font-size: 18px;
                    text-align: center;
                    letter-spacing: 3px;
                    font-family: "Courier New", monospace;
                    border: 1px solid #ddd;
                    border-radius: 4px;
                    box-sizing: border-box;
                }
                .s2fa-input:focus {
                    border-color: #0073aa;
                    box-shadow: 0 0 0 1px #0073aa;
                    outline: none;
                }
                .s2fa-toggle {
                    text-align: center;
                    margin: 16px 0;
                }
                .s2fa-toggle a {
                    color: #0073aa;
                    text-decoration: none;
                    font-size: 14px;
                }
                .s2fa-toggle a:hover {
                    text-decoration: underline;
                }
                .s2fa-backup-section {
                    display: none;
                }
                .s2fa-submit {
                    width: 100%;
                    padding: 12px 24px;
                    background: #0073aa;
                    color: white;
                    border: none;
                    border-radius: 4px;
                    font-size: 16px;
                    cursor: pointer;
                }
                .s2fa-submit:hover {
                    background: #005a87;
                }
                .s2fa-error {
                    background: #f8d7da;
                    color: #721c24;
                    padding: 12px;
                    border: 1px solid #f1aeb5;
                    border-radius: 4px;
                    margin-bottom: 20px;
                }
            </style>
        </head>
        <body class="login">
            <div id="login">
                <div class="s2fa-container">
                    <div class="s2fa-header">
                        <h1>Two-Factor Authentication</h1>
                        <p>Complete your login by entering your authentication code</p>
                    </div>
                    
                    <?php if (!empty($error_message)): ?>
                        <div class="s2fa-error">
                            <?php echo esc_html($error_message); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="s2fa-user-info">
                        <strong>Logging in as:</strong> <?php echo esc_html($user_login); ?><br>
                        <small style="color: #666;">
                            Session expires: 
                            <span id="session-timer">
                                <?php 
                                $remaining_minutes = ceil(($user_data['expires'] - time()) / 60);
                                echo $remaining_minutes . ' minutes';
                                ?>
                            </span>
                        </small>
                    </div>
                    
                    <form method="post" action="">
                        <div id="s2fa-totp-section" class="s2fa-form-section">
                            <label for="s2fa_code">
                                📱 Enter 6-digit code from your authenticator app:
                            </label>
                            <input type="text" 
                                   id="s2fa_code" 
                                   name="s2fa_code" 
                                   class="s2fa-input" 
                                   placeholder="000000" 
                                   maxlength="6" 
                                   autocomplete="one-time-code" 
                                   autofocus />
                        </div>
                        
                        <div class="s2fa-toggle">
                            <a href="#" id="s2fa-use-backup">Can't access your app? Use backup code</a>
                        </div>
                        
                        <div id="s2fa-backup-section" class="s2fa-backup-section s2fa-form-section">
                            <label for="s2fa_backup_code">
                                🔑 Enter backup code:
                            </label>
                            <input type="text" 
                                   id="s2fa_backup_code" 
                                   name="s2fa_backup_code" 
                                   class="s2fa-input" 
                                   placeholder="XXXX-XXXX" 
                                   maxlength="9"
                                   style="text-transform: uppercase;" />
                            <div class="s2fa-toggle">
                                <a href="#" id="s2fa-use-totp">← Back to authenticator app</a>
                            </div>
                        </div>
                        
                        <input type="hidden" name="s2fa_verify_submit" value="1" />
                        <input type="hidden" name="s2fa_token" value="<?php echo esc_attr($token); ?>" />
                        <button type="submit" class="s2fa-submit">Verify and Continue</button>
                    </form>
                </div>
            </div>
            
            <script>
                jQuery(document).ready(function($) {
                    // Auto-submit when 6-digit code is entered
                    $("#s2fa_code").on("input", function() {
                        $("#s2fa_backup_code").val(""); // Clear backup code
                        if (this.value.length === 6) {
                            setTimeout(function() {
                                $("form").submit();
                            }, 500);
                        }
                    });
                    
                    // Clear TOTP when backup code is entered
                    $("#s2fa_backup_code").on("input", function() {
                        $("#s2fa_code").val("");
                    });
                    
                    // Toggle between methods
                    $("#s2fa-use-backup").click(function(e) {
                        e.preventDefault();
                        $("#s2fa-totp-section").hide();
                        $("#s2fa-backup-section").show();
                        $("#s2fa_backup_code").focus();
                    });
                    
                    $("#s2fa-use-totp").click(function(e) {
                        e.preventDefault();
                        $("#s2fa-backup-section").hide();
                        $("#s2fa-totp-section").show();
                        $("#s2fa_code").focus();
                    });
                });
            </script>
            <?php wp_footer(); ?>
        </body>
        </html><?php
    }

    /**
     * Show 2FA verification form
     */
    public function show_2fa_form($user) {
        ?>
        <div class="s2fa-login-container">
            <div class="s2fa-header">
                <h3><?php _e('Two-Factor Authentication', 'secure-two-factor-auth'); ?></h3>
                <p class="s2fa-instruction">
                    <?php _e('Please enter your 6-digit code from your authenticator app.', 'secure-two-factor-auth'); ?>
                </p>
            </div>
            
            <div id="s2fa-totp-section" class="s2fa-section">
                <div class="s2fa-field-wrapper">
                    <label for="s2fa_code" class="s2fa-label">
                        <span class="s2fa-icon">📱</span>
                        <?php _e('Authenticator Code', 'secure-two-factor-auth'); ?>
                    </label>
                    <input type="text" 
                           id="s2fa_code" 
                           name="s2fa_code" 
                           class="s2fa-input" 
                           placeholder="000000" 
                           maxlength="6" 
                           autocomplete="one-time-code"
                           required />
                </div>
                
                <div class="s2fa-app-hints">
                    <small>
                        <?php _e('Open Microsoft Authenticator, Google Authenticator, or Authy', 'secure-two-factor-auth'); ?>
                    </small>
                </div>
            </div>
            
            <div id="s2fa-backup-code-section" class="s2fa-section" style="display: none;">
                <div class="s2fa-field-wrapper">
                    <label for="s2fa_backup_code" class="s2fa-label">
                        <span class="s2fa-icon">🔑</span>
                        <?php _e('Backup Code', 'secure-two-factor-auth'); ?>
                    </label>
                    <input type="text" 
                           id="s2fa_backup_code" 
                           name="s2fa_backup_code" 
                           class="s2fa-input" 
                           placeholder="XXXX-XXXX" 
                           maxlength="9"
                           style="text-transform: uppercase;" />
                </div>
                
                <div class="s2fa-backup-warning">
                    <small>
                        <?php _e('This backup code can only be used once.', 'secure-two-factor-auth'); ?>
                    </small>
                </div>
            </div>
            
            <div class="s2fa-options">
                <a href="#" id="s2fa-use-backup-code" class="s2fa-option-link">
                    <?php _e('Can\'t access your authenticator app? Use backup code', 'secure-two-factor-auth'); ?>
                </a>
                <a href="#" id="s2fa-use-totp-code" class="s2fa-option-link" style="display: none;">
                    <?php _e('← Back to authenticator app', 'secure-two-factor-auth'); ?>
                </a>
            </div>
        </div>
        <?php
    }
    
    /**
     * Enqueue login scripts
     */
    public function enqueue_login_scripts() {
        // No longer needed for inline form, kept for compatibility
    }
    
    /**
     * Handle successful login with 2FA
     */
    public function handle_2fa_login($user_login, $user) {
        if (S2FA_User::is_2fa_enabled($user->ID)) {
            error_log("Successful 2FA login for user: $user_login (ID: {$user->ID})");
            do_action('s2fa_successful_login', $user->ID);
        }
    }
    
    /**
     * Verify TOTP code
     */
    private function verify_totp_code($user_id, $code) {
        $secret = S2FA_User::get_user_secret($user_id);
        
        if (!$secret) {
            return false;
        }
        
        return S2FA_TOTP::verify_code($code, $secret);
    }
    
    /**
     * Verify backup code
     */
    private function verify_backup_code($user_id, $code) {
        return S2FA_User::use_backup_code($user_id, $code);
    }
    
    /**
     * Show thank you page with 4-second delay before redirect
     */
    private function show_thank_you_page($redirect_to) {
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta http-equiv="Content-Type" content="<?php bloginfo('html_type'); ?>; charset=<?php echo get_option('blog_charset'); ?>" />
            <title><?php echo get_bloginfo('name', 'display'); ?> &rsaquo; Login Successful</title>
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    margin: 0;
                    padding: 0;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    min-height: 100vh;
                    color: white;
                }
                .thank-you-container {
                    text-align: center;
                    background: rgba(255, 255, 255, 0.1);
                    padding: 40px;
                    border-radius: 15px;
                    backdrop-filter: blur(10px);
                    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
                    max-width: 400px;
                    width: 90%;
                }
                .success-icon {
                    font-size: 60px;
                    color: #4CAF50;
                    margin-bottom: 20px;
                    animation: bounce 1s ease-in-out;
                }
                .thank-you-title {
                    font-size: 28px;
                    font-weight: 600;
                    margin-bottom: 15px;
                }
                .thank-you-message {
                    font-size: 16px;
                    opacity: 0.9;
                    margin-bottom: 20px;
                }
                .countdown {
                    font-size: 18px;
                    font-weight: bold;
                    color: #4CAF50;
                }
                .loader {
                    border: 3px solid rgba(255,255,255,0.3);
                    border-radius: 50%;
                    border-top: 3px solid #4CAF50;
                    width: 30px;
                    height: 30px;
                    animation: spin 1s linear infinite;
                    margin: 20px auto;
                }
                @keyframes bounce {
                    0%, 20%, 60%, 100% { transform: translateY(0); }
                    40% { transform: translateY(-20px); }
                    80% { transform: translateY(-10px); }
                }
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            </style>
        </head>
        <body>
            <div class="thank-you-container">
                <div class="success-icon">✅</div>
                <div class="thank-you-title">Authentication Successful!</div>
                <div class="thank-you-message">Your two-factor authentication was verified successfully.</div>
                <div class="loader"></div>
                <div class="countdown">Redirecting in <span id="countdown">4</span> seconds...</div>
            </div>

            <script>
                let timeLeft = 4;
                const countdownElement = document.getElementById('countdown');
                
                const timer = setInterval(function() {
                    timeLeft--;
                    countdownElement.textContent = timeLeft;
                    
                    if (timeLeft <= 0) {
                        clearInterval(timer);
                        window.location.replace('<?php echo esc_js($redirect_to); ?>');
                    }
                }, 1000);
                
                // Also set a failsafe redirect
                setTimeout(function() {
                    window.location.replace('<?php echo esc_js($redirect_to); ?>');
                }, 4500);
            </script>
        </body>
        </html>
        <?php
    }
}