<?php
/**
 * Secure Two Factor Authentication - TOTP Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class S2FA_TOTP {
    
    /**
     * Generate a random secret key
     */
    public static function generate_secret($length = 32) {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $secret = '';
        
        for ($i = 0; $i < $length; $i++) {
            $secret .= $chars[random_int(0, strlen($chars) - 1)];
        }
        
        return $secret;
    }
    
    /**
     * Verify TOTP code
     */
    public static function verify_code($code, $secret, $window = 2) {
        if (empty($code) || empty($secret)) {
            return false;
        }
        
        // Clean the code
        $code = preg_replace('/[^0-9]/', '', $code);
        
        if (strlen($code) !== 6) {
            return false;
        }
        
        $time = time();
        $period = 30; // 30 second window
        
        // Check current time and surrounding windows
        for ($i = -$window; $i <= $window; $i++) {
            $timestamp = $time + ($i * $period);
            $expected_code = self::generate_code($secret, floor($timestamp / $period));
            
            if (hash_equals($code, $expected_code)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Generate TOTP code for a given time
     */
    public static function generate_code($secret, $time = null) {
        if ($time === null) {
            $time = floor(time() / 30);
        }
        
        // Convert secret from base32
        $secret_binary = self::base32_decode($secret);
        
        if (!$secret_binary) {
            return false;
        }
        
        // Convert time to binary
        $time_binary = pack('N*', 0, $time);
        
        // Generate HMAC hash
        $hash = hash_hmac('sha1', $time_binary, $secret_binary, true);
        
        // Dynamic truncation
        $offset = ord($hash[19]) & 0xf;
        $code = (
            ((ord($hash[$offset]) & 0x7f) << 24) |
            ((ord($hash[$offset + 1]) & 0xff) << 16) |
            ((ord($hash[$offset + 2]) & 0xff) << 8) |
            (ord($hash[$offset + 3]) & 0xff)
        ) % pow(10, 6);
        
        return str_pad($code, 6, '0', STR_PAD_LEFT);
    }
    
    /**
     * Base32 decode function
     */
    private static function base32_decode($input) {
        if (empty($input)) {
            return false;
        }
        
        $input = strtoupper($input);
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $output = '';
        $v = 0;
        $vbits = 0;
        
        for ($i = 0, $j = strlen($input); $i < $j; $i++) {
            $v <<= 5;
            $pos = strpos($alphabet, $input[$i]);
            if ($pos === false) {
                continue; // Skip invalid characters
            }
            $v += $pos;
            $vbits += 5;
            
            if ($vbits >= 8) {
                $output .= chr(($v >> ($vbits - 8)) & 255);
                $vbits -= 8;
            }
        }
        
        return $output;
    }
    
    /**
     * Base32 encode function
     */
    private static function base32_encode($input) {
        if (empty($input)) {
            return '';
        }
        
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $output = '';
        $v = 0;
        $vbits = 0;
        
        for ($i = 0, $j = strlen($input); $i < $j; $i++) {
            $v <<= 8;
            $v += ord($input[$i]);
            $vbits += 8;
            
            while ($vbits >= 5) {
                $output .= $alphabet[($v >> ($vbits - 5)) & 31];
                $vbits -= 5;
            }
        }
        
        if ($vbits > 0) {
            $output .= $alphabet[($v << (5 - $vbits)) & 31];
        }
        
        return $output;
    }
    
    /**
     * Get current TOTP code for testing purposes
     */
    public static function get_current_code($secret) {
        return self::generate_code($secret);
    }
    
    /**
     * Get remaining time for current TOTP code
     */
    public static function get_remaining_time() {
        return 30 - (time() % 30);
    }
    
    /**
     * Validate secret key format
     */
    public static function is_valid_secret($secret) {
        if (empty($secret)) {
            return false;
        }
        
        // Check if it's valid base32
        return preg_match('/^[A-Z2-7]+$/', $secret) && strlen($secret) >= 16;
    }
}