<?php
/**
 * Secure Two Factor Authentication - User Management
 */

if (!defined('ABSPATH')) {
    exit;
}

class S2FA_User {
    
    /**
     * Check if 2FA is enabled for a user
     */
    public static function is_2fa_enabled($user_id) {
        $enabled = get_user_meta($user_id, 's2fa_enabled', true);
        return !empty($enabled) && $enabled === '1';
    }
    
    /**
     * Enable 2FA for a user
     */
    public static function enable_2fa($user_id) {
        return update_user_meta($user_id, 's2fa_enabled', '1');
    }
    
    /**
     * Disable 2FA for a user
     */
    public static function disable_2fa($user_id) {
        delete_user_meta($user_id, 's2fa_enabled');
        delete_user_meta($user_id, 's2fa_secret');
        delete_user_meta($user_id, 's2fa_backup_codes');
        self::clear_backup_codes($user_id);
        return true;
    }
    
    /**
     * Get user's 2FA secret
     */
    public static function get_user_secret($user_id) {
        return get_user_meta($user_id, 's2fa_secret', true);
    }
    
    /**
     * Set user's 2FA secret
     */
    public static function set_user_secret($user_id, $secret) {
        return update_user_meta($user_id, 's2fa_secret', $secret);
    }
    
    /**
     * Generate a new secret for a user
     */
    public static function generate_secret($user_id) {
        $secret = S2FA_TOTP::generate_secret();
        self::set_user_secret($user_id, $secret);
        return $secret;
    }
    
    /**
     * Generate backup codes for a user
     */
    public static function generate_backup_codes($user_id, $count = 10) {
        global $wpdb;
        
        // Clear existing backup codes
        self::clear_backup_codes($user_id);
        
        $codes = array();
        $table_name = $wpdb->prefix . 's2fa_backup_codes';
        
        for ($i = 0; $i < $count; $i++) {
            $code = self::generate_backup_code();
            $codes[] = $code;
            
            $wpdb->insert(
                $table_name,
                array(
                    'user_id' => $user_id,
                    'code' => $code,
                    'used' => 0,
                    'created_at' => current_time('mysql')
                ),
                array('%d', '%s', '%d', '%s')
            );
        }
        
        return $codes;
    }
    
    /**
     * Generate a single backup code
     */
    private static function generate_backup_code() {
        $part1 = str_pad(mt_rand(0, 9999), 4, '0', STR_PAD_LEFT);
        $part2 = str_pad(mt_rand(0, 9999), 4, '0', STR_PAD_LEFT);
        return $part1 . '-' . $part2;
    }
    
    /**
     * Get unused backup codes for a user
     */
    public static function get_backup_codes($user_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 's2fa_backup_codes';
        
        $codes = $wpdb->get_results($wpdb->prepare(
            "SELECT code FROM $table_name WHERE user_id = %d AND used = 0 ORDER BY created_at ASC",
            $user_id
        ));
        
        return array_column($codes, 'code');
    }
    
    /**
     * Use a backup code (mark as used)
     */
    public static function use_backup_code($user_id, $code) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 's2fa_backup_codes';
        
        // Check if code exists and is unused
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table_name WHERE user_id = %d AND code = %s AND used = 0",
            $user_id,
            $code
        ));
        
        if (!$existing) {
            return false;
        }
        
        // Mark code as used
        $updated = $wpdb->update(
            $table_name,
            array('used' => 1),
            array('id' => $existing->id),
            array('%d'),
            array('%d')
        );
        
        return $updated !== false;
    }
    
    /**
     * Clear all backup codes for a user
     */
    public static function clear_backup_codes($user_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 's2fa_backup_codes';
        
        return $wpdb->delete(
            $table_name,
            array('user_id' => $user_id),
            array('%d')
        );
    }
    
    /**
     * Get QR code URL for Google Authenticator setup
     */
    public static function get_qr_code_url($user_id, $secret = null) {
        if (!$secret) {
            $secret = self::get_user_secret($user_id);
        }
        
        if (!$secret) {
            return false;
        }
        
        $user = get_user_by('id', $user_id);
        if (!$user) {
            return false;
        }
        
        $site_name = get_bloginfo('name');
        $user_email = $user->user_email;
        
        $qr_data = "otpauth://totp/" . urlencode($site_name . ":" . $user_email) . "?secret=" . $secret . "&issuer=" . urlencode($site_name);
        
        // Try multiple QR code services
        $qr_services = array(
            "https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=" . urlencode($qr_data),
            "https://chart.googleapis.com/chart?chs=200x200&chld=M|0&cht=qr&chl=" . urlencode($qr_data),
            "https://qr-server.com/api/v1/create-qr-code/?size=200x200&data=" . urlencode($qr_data)
        );
        
        // Return the first available service (we'll handle fallbacks in JavaScript)
        return $qr_services[0];
    }
    
    /**
     * Get setup instructions for the user
     */
    public static function get_setup_instructions($user_id) {
        $secret = self::get_user_secret($user_id);
        if (!$secret) {
            return false;
        }
        
        return array(
            'secret' => $secret,
            'qr_url' => self::get_qr_code_url($user_id, $secret),
            'manual_entry' => chunk_split($secret, 4, ' '),
            'instructions' => array(
                __('Install an authenticator app like Google Authenticator, Microsoft Authenticator, or Authy on your mobile device.', 'secure-two-factor-auth'),
                __('Scan the QR code below with your authenticator app, or manually enter the secret key.', 'secure-two-factor-auth'),
                __('Enter the 6-digit code from your app to verify the setup.', 'secure-two-factor-auth'),
                __('Save your backup codes in a safe place. You can use them if you lose access to your authenticator app.', 'secure-two-factor-auth')
            )
        );
    }
}